<?php

namespace App\Services\Xml;

use Illuminate\Support\Facades\Validator;

class ValidadorLayoutXmlRequest
{
    /**
     * Responsável pela validação dos valores no array php, gerando mensagemns de erros, caso existam os erros.
     * @param array $data
     * @return array|bool
     */
    public static function validar(array $data)
    {
        $cabecalhoValidator = Validator::make($data, self::cabecalhoRules(), self::cabecalhoMessages());

        if (count($cabecalhoValidator->errors()) > 0)
            return $cabecalhoValidator->errors()->messages();

        foreach ($data['rpsDoLote'] as $rps) {

            $rpsValidator = Validator::make($rps, self::rpsRules(), self::rpsMessages());

            if (count($rpsValidator->errors()) > 0) {
                return $rpsValidator->errors()->messages();
            }
        }

        return false;
    }

    /**
     * ------------------------------------------
     * Validação apenas para o cabeçalho do lote.
     * ------------------------------------------
     */

    private static function cabecalhoRules() : array
    {
        return [
            "LoteRpsId" => 'required',
            "NumeroLote" => 'required|numeric',
            "Cnpj" => 'required|cnpj',
            "InscricaoMunicipal" => 'required',
            "CodigoCidadeCadastroPrestador" => 'required',
        ];
    }

    private static function cabecalhoMessages() : array
    {
        return [
            'LoteRpsId.required' => 'O ID do Lote não pode ser NULL',

            'NumeroLote.required' => 'Erro CÓD: 104 - Número do Lote RPS inválido: não pode ser NULL',
            'NumeroLote.numeric' => 'Erro CÓD: 104 - Número do Lote inválido: deve ser um valor numérico',

            'Cnpj.required' => 'Erro CÓD: 302 - CNPJ inválido: Forneça um CNPJ válido',
            'Cnpj.cnpj' => 'Erro CÓD: 302 - CNPJ inválido: Forneça um CNPJ válido',

            'InscricaoMunicipal.required' => 'Erro CÓD: 103 - Inscrição inválido: Informar a inscrição municipal do contribuinte',

            'CodigoCidadeCadastroPrestador.required' => 'Código da Cidade/Município do Prestador Inválido: Utilizar códigos dos municípios fornecidos pelo IBGE, disponíveis em https://www.ibge.gov.br/explica/codigos-dos-municipios.php#PB',
        ];
    }

    /**
     * -----------------------------------------
     * Validação para o corpo da requisição.
     * Ou seja, para os atributos das RPS.
     * -----------------------------------------
     */

    private static function rpsRules() : array
    {
        return [
            'Id'                        => 'required|numeric',
            'NumeroRps'                 => 'required|numeric',
            //'Serie'                     => '',
            'Tipo'                      => 'required|string',
            'DataEmissaoRps'            => 'required|date',
            'NaturezaOperacao'          => 'required',
            'RegimeEspecialTributacao'  => 'required',
            'OptanteSimplesNacional'    => 'required',
            'IncentivadorCultural'      => 'required',
            'Competencia'               => 'required',
            'Status'                    => 'required',

                //Serviços - Valores
            'ValorServicos'             => 'required',
            'ValorDeducoes'             => 'required',
            'ValorPis'                  => 'required',
            'ValorCofins'               => 'required',
            'ValorInss'                 => 'required',
            'ValorIr'                   => 'required',
            'ValorCsll'                 => 'required',
            'ValorIssRetido'            => 'required',

            'RetencaoIss'               => 'required|min:1|numeric|between: 1,2',

            'ValorIss'                  => 'required',
            'OutrasRetencoes'           => 'required',
            'Aliquota'                  => 'required',
            'DescontoIncondicionado'    => 'required',
            'DescontoCondicionado'      => 'required',

                //Serviços
            'ItemListaServico'          => 'required|string',
            'CodigoCnae'                => 'required',
            'CodigoTributacaoMunicipio' => 'required',
            'Discriminacao'             => 'required',
            'CodigoMunicipioServico'    => 'required',

                //Tomador
            'CpfCnpjTomador'            => 'required|cpf_ou_cnpj',
            'RazaoSocialTomador'               => 'required',
            'EnderecoTomador'                  => 'required',
            'NumeroEnderecoTomador'     => 'required',
            //'ComplementoTomador'               => 'required',
            'BairroTomador'                    => 'required',
            'CodigoMunicipioTomador'    => 'required',
            'UfTomador'                        => 'required|uf',
            'CepTomador'                       => 'required',

            //Contatos
            //'TelefoneTomador'                  => 'required',
            //'EmailTomador'                     => 'required',
        ];
    }

    private static function rpsMessages() : array
    {
        return [
            'Id.required'                       => 'O ID da RPS não pode ser NULL',
            'Id.numeric'                        => 'O ID da RPS deve ser um valor numérico',

            'NumeroRps.required'                => 'Erro CÓD: 204 - Número de RPS inválido: Forneça um número de RPS válido',
            'NumeroRps.numeric'                 => 'Erro CÓD: 204 - Número de RPS inválido: Forneça um número de RPS válido',

            'Serie.required'                   => 'O campo Série é obrigatório',

            'Tipo.required'                     => 'Erro CÓD: 202 - Tipo RPS inválido: Informe RPS',
            'Tipo.string'                       => 'Erro CÓD: 202 - Tipo RPS inválido: Informe RPS',

            'DataEmissaoRps.required'           => 'Erro CÓD: 207 - Data da RPS inválida: Informe uma data válida',

            'NaturezaOperacao.required'         => 'O campo natureza da operação é obrigatória',
            'RegimeEspecialTributacao.required' => 'O campo Regime Especial Tributação é obrigatório',
            'OptanteSimplesNacional.required'   => 'O campo Optante Simples Nacional obrigatório',
            'IncentivadorCultural.required'     => 'O campo Incentivador Cultural é obrigatório',
            'Competencia.required'              => 'O Campo Competência é obrigatório',
            'Status.required'                   => 'O Status da operação é obrigatório',

            //Serviços - Valores
            'ValorServicos.required'            => 'Erro CÓD: 405 - Valor da RPS (Valor Serviço) inválido: Informe um valor válido',
            'ValorDeducoes.required'            => 'Erro CÓD: 406 - Valor da dedução inválido: Informe um valor válido',
            'ValorPis.required'                 => 'Erro CÓD: 416 - Valor do PIS inválido: Informe um valor válido',
            'ValorCofins.required'              => 'Erro CÓD: 412 - Valor do COFINS inválido: Informe um valor válido',
            'ValorInss.required'                => 'Erro CÓD: 414 - Valor do INSS inválido: Informe um valor válido',
            'ValorIr.required'                  => 'Erro CÓD: 415 - Valor do IR inválido: Informe um valor válido',
            'ValorCsll.required'                => 'Erro CÓD: 413 - Valor do CSLL inválido: Informe um valor válido',

            'ValorIssRetido.required'           => 'Erro CÓD: 417 - Valor do ISS RETIDO inválido: Informe um valor válido',
            'ValorIss.required'                 => 'Erro CÓD: 417 - Valor do ISS inválido: Informe um valor válido',

            'RetencaoIss.required'              => 'Erro: A retenção do ISS é obrigatório. Use os valores (1) para SIM; (2) para NÃO.',
            'RetencaoIss.min'                   => 'Erro: A retenção do ISS precisa ter um valor entre 1 e 2. Use os valores (1) para SIM; (2) para NÃO.',
            'RetencaoIss.between'               => 'Erro: Informe se há retenção de ISS: Use os valores (1) para SIM; (2) para NÃO.',
            'RetencaoIss.numeric'               => 'Erro: A retenção de ISS deve ser um inteiro: Use os valores (1) para SIM; (2) para NÃO.',

            'OutrasRetencoes.required'          => 'Erro CÓD: 417 - Valor de OUTRAS RETENÇÕES inválido: Informe um valor válido',
            'Aliquota.required'                 => 'Erro CÓD: 408 - Valor da Alíquota inválido: Informe um valor válido',
            'DescontoIncondicionado.required'   => 'Valor Desconto Incondicionado inválido: Informe um valor válido',
            'DescontoCondicionado.required'     => 'Valor Desconto Condicionado inválido: Informe um valor válido',

            //Serviços
            'ItemListaServico.required'         => 'Item Lista de Serviço não pode ser NULL',
            'CodigoCnae.required'               => 'Código CNAE não pode ser NULL',
            'CodigoTributacaoMunicipio.required'=> 'Erro CÓD: 402 - Tributação inválida: Forneça um dos tipos especificados',
            'Discriminacao.required'            => 'Descriminação do Serviço não pode ser NULL',
            'CodigoMunicipioServico.required'   => 'Erro CÓD: 410 - Código da Cidade da Prestação de Serviço inválido: Forneça código válido',

            //Tomador
            'CpfCnpjTomador.required'           => 'Erro CÓD: 302 - CPF/CNPJ Tomador é obrigatório',
            'CpfCnpjTomador.cpf_ou_cnpj'        => 'Erro CÓD: 302 - CPF/CNPJ Tomador inválido: Forneça um CPF/CNPJ válido',

            'RazaoSocialTomador.required'       => 'Erro CÓD: 304 - Razão Social/Nome Tomador inválido: Forneça a Razão Social/Nome do Tomador do serviço',
            'EnderecoTomador.required'          => 'Erro CÓD: 309 - Logradouro do Tomador de Serviço inválido: Forneça o Logradouro do Tomador do serviço',
            'NumeroEnderecoTomador.required'    => 'Erro CÓD: 310 - Número do logradouro do Tomador de Serviço inválido: Forneça o Número logradouro do Tomador do serviço',
            //'ComplementoTomador'       => '',
            'BairroTomador.required'            => 'Erro CÓD: 311 - Bairro do Tomador de Serviço inválido: Forneça o Bairro do Tomador do serviço',
            'CodigoMunicipioTomador.required'   => 'Erro CÓD: 312 - Cidade inválida: Forneça a cidade do Tomador do serviço',
            'UfTomador.required'                => 'Erro CÓD: 313 - UF do Tomador de Serviço inválido: Forneça UF do Tomador do serviço',
            'UfTomador.uf'                      => 'Erro CÓD: 313 - UF do Tomador de Serviço inválido: Forneça UF do Tomador do serviço',
            'CepTomador.required'               => 'Erro CÓD: 314 - CEP inválido: Forneça o CPF do Tomador do serviço',

            //Contato
            //'TelefoneTomador.required'          => 'Erro CÓD: 314 - CEP inválido: Forneça o CPF do Tomador do serviço',
            //'EmailTomador.required'             => 'Erro CÓD: 314 - CEP inválido: Forneça o CPF do Tomador do serviço',
        ];
    }
}
