<?php

namespace App\Services\Xml;

use App\Services\Instituicao\BuscaInstituicao;
use Exception;

class FormatadorRps
{
    /**
     * Formata os dados recebido no arquivo xml que já foram convertidos em array php.
     * Também simplifica a estrutura dos dados.
     * Exemplo:
     *  XML: acesso ao Id do dado: $lote['LoteRps']['@attributes']['Id'];
     *  ARRAY PHP: acesso ao Id formatado: $loteRps['LoteRpsId'];
     *
     * @param array $lote
     * @return array
     * @throws Exception
     */
    public static function formatar(array $lote) : array
    {
        if (is_null($lote) || count($lote) < 1)
            throw new Exception("A RPS passada é nula ou está vazia.");

        try {

            if (empty($lote['LoteRps']['CodigoCidadeCadastroPrestador']))
                throw new Exception("Código da Cidade/Município do Prestador Inválido: Verifique o Manual do WebService para checar a estrutura do XML e utilize os códigos dos municípios fornecidos pelo IBGE, disponíveis em https://www.ibge.gov.br/explica/codigos-dos-municipios.php#PB");

            $codigoCidade = $lote['LoteRps']['CodigoCidadeCadastroPrestador'];

            $instituicao = BuscaInstituicao::buscarPorCodigoDaCidade($codigoCidade);

            if (!$instituicao)
                throw new Exception("O código da cidade do Prestador não corresponde a nenhuma cidade cadastrada em nosso sistema.
                 Por favor, verifique o código correto em: https://www.ibge.gov.br/explica/codigos-dos-municipios.php#PB");

            $loteRps = [];

            $loteRps['LoteRpsId']                       = $lote['LoteRps']['@attributes']['Id'];
            $loteRps['NumeroLote']                      = $lote['LoteRps']['NumeroLote'];
            $loteRps['InstituicaoCnpj']                 = $instituicao->cnpj;
            $loteRps['Cnpj']                            = $lote['LoteRps']['Cnpj'];
            $loteRps['InscricaoMunicipal']              = $lote['LoteRps']['InscricaoMunicipal'];
            $loteRps['CodigoCidadeCadastroPrestador']   = $lote['LoteRps']['CodigoCidadeCadastroPrestador'];
            $loteRps['QuantidadeRps']                   = $lote['LoteRps']['QuantidadeRps'];

            if (count($lote['LoteRps']['ListaRps']['Rps']) > 1) {
                $loteRps['rpsDoLote'] = (new FormataListaRpsCom2OuMais)->formatar($lote);
            } else {
                $loteRps['rpsDoLote'] = (new FormataListaRpsComApenas1)->formatar($lote);
            }

            return $loteRps;

        } catch (Exception $exception) {
            throw new Exception($exception->getMessage());
        }

    }
}
